<?php
/**
 * PayTech (Intech) - WHMCS Third Party Gateway Module
 *
 * - Creates a PayTech payment request (request-payment) and redirects the client to PayTech checkout
 * - Receives IPN notifications in /modules/gateways/callback/intechpaytech.php
 *
 * Docs:
 * - PayTech request-payment endpoint + headers API_KEY/API_SECRET and form encoding
 * - IPN verification supports HMAC-SHA256 or SHA256 hashed keys
 *
 * Compatible with WHMCS 8.x (Third Party Gateway).
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

function intechpaytech_MetaData()
{
    return array(
        'DisplayName' => 'PayTech (Intech) - Redirect Checkout',
        'APIVersion' => '1.1',
        'DisableLocalCreditCardInput' => true,
        'TokenisedStorage' => false,
    );
}

function intechpaytech_config()
{
    return array(
        'FriendlyName' => array(
            'Type'  => 'System',
            'Value' => 'PayTech (Intech) - Redirect Checkout',
        ),
        'apiKey' => array(
            'FriendlyName' => 'API Key',
            'Type'         => 'text',
            'Size'         => '50',
            'Default'      => '',
            'Description'  => 'Votre clé API PayTech (API_KEY)',
        ),
        'apiSecret' => array(
            'FriendlyName' => 'API Secret',
            'Type'         => 'password',
            'Size'         => '50',
            'Default'      => '',
            'Description'  => 'Votre clé secrète PayTech (API_SECRET)',
        ),
        'environment' => array(
            'FriendlyName' => 'Environnement PayTech',
            'Type'         => 'dropdown',
            'Options'      => array(
                'prod' => 'Production',
                'test' => 'Sandbox/Test',
            ),
            'Description'  => 'Choisir prod ou test. Le sandbox sert aux tests internes uniquement.',
        ),
        'targetPayment' => array(
            'FriendlyName' => 'Méthodes ciblées (target_payment)',
            'Type'         => 'text',
            'Size'         => '50',
            'Default'      => 'Orange Money, Wave, Free Money',
            'Description'  => 'Ex: "Orange Money" ou "Orange Money, Wave, Free Money". Laisser vide pour afficher tout.',
        ),
        'popupMode' => array(
            'FriendlyName' => 'Mode Pop-up (redirection)',
            'Type'         => 'yesno',
            'Description'  => 'Si coché, ouvre PayTech dans une pop-up (sinon redirection normale).',
        ),
        'debugLog' => array(
            'FriendlyName' => 'Debug Log (Gateway Log)',
            'Type'         => 'yesno',
            'Description'  => 'Si coché, log plus de détails (réponses API, erreurs).',
        ),
    );
}

function intechpaytech_link($params)
{
    $apiKey       = trim($params['apiKey'] ?? '');
    $apiSecret    = trim($params['apiSecret'] ?? '');
    $environment  = ($params['environment'] ?? 'prod') === 'test' ? 'test' : 'prod';
    $targetPayment = trim($params['targetPayment'] ?? '');
    $popupMode    = !empty($params['popupMode']);
    $debugLog     = !empty($params['debugLog']);

    $invoiceId    = (int)($params['invoiceid'] ?? 0);
    $amount       = $params['amount'] ?? '0.00';
    $currency     = strtoupper($params['currency'] ?? 'XOF');
    $description  = $params['description'] ?? ('Invoice #' . $invoiceId);
    $companyName  = $params['companyname'] ?? 'WHMCS';
    $systemUrl    = rtrim($params['systemurl'] ?? '', '/');
    $returnUrl    = $params['returnurl'] ?? $systemUrl . '/viewinvoice.php?id=' . $invoiceId;

    // Client details (optional; may be used by PayTech in checkout)
    $client = $params['clientdetails'] ?? array();
    $firstname = $client['firstname'] ?? '';
    $lastname  = $client['lastname'] ?? '';
    $fullname  = trim($firstname . ' ' . $lastname);
    $phone     = $client['phonenumber'] ?? '';

    if ($apiKey === '' || $apiSecret === '' || $invoiceId <= 0) {
        return '<div style="color:#b00;">PayTech gateway: configuration incomplète (API Key/Secret) ou facture invalide.</div>';
    }

    // PayTech supports: XOF, EUR, USD, CAD, GBP, MAD (if currency not supported, fallback to XOF)
    $supported = array('XOF','EUR','USD','CAD','GBP','MAD');
    if (!in_array($currency, $supported, true)) {
        $currency = 'XOF';
    }

    // Amount formatting: for XOF/MAD, usually integer; keep safe rounding
    $amountNumeric = (float)$amount;
    if ($currency === 'XOF' || $currency === 'MAD') {
        $amountNumeric = (float)round($amountNumeric, 0);
    } else {
        $amountNumeric = (float)round($amountNumeric, 2);
    }

    // Build PayTech request payload
    $refCommand = 'WHMCS-' . $invoiceId . '-' . time();
    $customField = json_encode(array(
        'invoice_id' => $invoiceId,
        'client_id'  => (int)($client['id'] ?? 0),
        'currency'   => $currency,
        'amount'     => $amountNumeric,
    ));

    $ipnUrl = $systemUrl . '/modules/gateways/callback/intechpaytech.php';
    $successUrl = $returnUrl; // after payment, WHMCS returnurl points to invoice/complete
    $cancelUrl  = $returnUrl;

    $postFields = array(
        'item_name'     => $companyName,
        'item_price'    => $amountNumeric,
        'ref_command'   => $refCommand,
        'command_name'  => $description,
        'currency'      => $currency,
        'env'           => $environment,
        'ipn_url'       => $ipnUrl,
        'success_url'   => $successUrl,
        'cancel_url'    => $cancelUrl,
        'custom_field'  => $customField,
    );
    if ($targetPayment !== '') {
        $postFields['target_payment'] = $targetPayment;
    }

    $apiUrl = 'https://paytech.sn/api/payment/request-payment';

    // Call PayTech API (form-encoded)
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postFields));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'API_KEY: ' . $apiKey,
        'API_SECRET: ' . $apiSecret,
        'Content-Type: application/x-www-form-urlencoded'
    ));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);

    $response = curl_exec($ch);
    $curlErr  = curl_error($ch);
    $httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($response === false || $curlErr) {
        if ($debugLog && function_exists('logTransaction')) {
            logTransaction('intechpaytech', array('error' => $curlErr, 'httpCode' => $httpCode), 'Error');
        }
        return '<div style="color:#b00;">Erreur PayTech: impossible de contacter l’API. Merci de réessayer.</div>';
    }

    $data = json_decode($response, true);

    if ($debugLog && function_exists('logTransaction')) {
        logTransaction('intechpaytech', array('httpCode' => $httpCode, 'response' => $data, 'raw' => $response), 'Debug');
    }

    if (!is_array($data) || empty($data['success']) || empty($data['redirect_url'])) {
        $msg = 'Réponse PayTech invalide.';
        if (is_array($data) && isset($data['message'])) {
            $msg = (string)$data['message'];
        }
        return '<div style="color:#b00;">PayTech: ' . htmlspecialchars($msg) . '</div>';
    }

    $redirectUrl = $data['redirect_url'];

    // Popup mode: open PayTech checkout in a centered popup; also provide fallback link
    if ($popupMode) {
        $btn = htmlspecialchars($params['langpaynow'] ?? 'Pay Now');
        $safeUrl = htmlspecialchars($redirectUrl);
        return <<<HTML
<div>
  <button type="button" onclick="intechPaytechOpenPopup()">{$btn}</button>
  <noscript><p><a href="{$safeUrl}" target="_blank" rel="noopener">Ouvrir la page de paiement</a></p></noscript>
</div>
<script>
function intechPaytechOpenPopup() {
  var w = 520, h = 740;
  var dualScreenLeft = window.screenLeft !== undefined ? window.screenLeft : window.screenX;
  var dualScreenTop = window.screenTop !== undefined ? window.screenTop : window.screenY;
  var width = window.innerWidth ? window.innerWidth : document.documentElement.clientWidth ? document.documentElement.clientWidth : screen.width;
  var height = window.innerHeight ? window.innerHeight : document.documentElement.clientHeight ? document.documentElement.clientHeight : screen.height;
  var left = ((width / 2) - (w / 2)) + dualScreenLeft;
  var top = ((height / 2) - (h / 2)) + dualScreenTop;
  var win = window.open("{$safeUrl}", "paytechCheckout", "scrollbars=yes, width=" + w + ", height=" + h + ", top=" + top + ", left=" + left);
  if (win && win.focus) win.focus();
  if (!win) {
    window.location.href = "{$safeUrl}";
  }
}
</script>
HTML;
    }

    // Standard redirect: show a "Pay Now" button linking to checkout
    $btn = htmlspecialchars($params['langpaynow'] ?? 'Pay Now');
    $safeUrl = htmlspecialchars($redirectUrl);
    return '<a class="btn btn-primary" href="' . $safeUrl . '" rel="noopener">' . $btn . '</a>';
}
