<?php
/**
 * PayTech (Intech) - WHMCS Callback/IPN handler
 *
 * Receives PayTech IPN POST requests and applies invoice payments in WHMCS.
 *
 * Verification:
 * - Preferred: compare received hmac_compute with locally computed HMAC-SHA256:
 *   message = item_price|ref_command|api_key
 * - Fallback: compare api_key_sha256 and api_secret_sha256 to sha256(api_key/secret)
 */

use WHMCS\Database\Capsule;

require_once __DIR__ . '/../../../../init.php';
require_once __DIR__ . '/../../../../includes/gatewayfunctions.php';
require_once __DIR__ . '/../../../../includes/invoicefunctions.php';

$gatewayModuleName = 'intechpaytech';
$gatewayParams = getGatewayVariables($gatewayModuleName);

if (!$gatewayParams['type']) {
    // Module not active
    http_response_code(400);
    echo "Module Not Activated";
    exit;
}

$apiKey    = trim($gatewayParams['apiKey'] ?? '');
$apiSecret = trim($gatewayParams['apiSecret'] ?? '');
$debugLog  = !empty($gatewayParams['debugLog']);

$input = $_POST;
$event = $input['type_event'] ?? '';
$refCommand = $input['ref_command'] ?? '';
$itemPrice  = $input['item_price'] ?? '';
$token      = $input['token'] ?? ($input['token_payment'] ?? '');
$hmacReceived = $input['hmac_compute'] ?? '';
$keyHashReceived = $input['api_key_sha256'] ?? '';
$secretHashReceived = $input['api_secret_sha256'] ?? '';

// Helper: compute HMAC
function intechpaytech_compute_hmac($amount, $refCommand, $apiKey, $apiSecret) {
    $message = $amount . '|' . $refCommand . '|' . $apiKey;
    return hash_hmac('sha256', $message, $apiSecret);
}

function intechpaytech_compute_sha256($value) {
    return hash('sha256', $value);
}

$valid = false;

// Method 1: HMAC-SHA256 (recommended)
if ($hmacReceived !== '' && $apiKey !== '' && $apiSecret !== '' && $refCommand !== '') {
    $expected = intechpaytech_compute_hmac($itemPrice, $refCommand, $apiKey, $apiSecret);
    // timing-safe compare
    if (function_exists('hash_equals')) {
        $valid = hash_equals($expected, $hmacReceived);
    } else {
        $valid = ($expected === $hmacReceived);
    }
}

// Method 2: SHA256 hashes (fallback)
if (!$valid && $keyHashReceived !== '' && $secretHashReceived !== '' && $apiKey !== '' && $apiSecret !== '') {
    $expectedKeyHash = intechpaytech_compute_sha256($apiKey);
    $expectedSecretHash = intechpaytech_compute_sha256($apiSecret);
    if (function_exists('hash_equals')) {
        $valid = hash_equals($expectedKeyHash, $keyHashReceived) && hash_equals($expectedSecretHash, $secretHashReceived);
    } else {
        $valid = ($expectedKeyHash === $keyHashReceived) && ($expectedSecretHash === $secretHashReceived);
    }
}

if (!$valid) {
    logTransaction($gatewayModuleName, array('input' => $input, 'reason' => 'Invalid signature'), 'Invalid');
    http_response_code(403);
    echo "IPN KO";
    exit;
}

// Decode custom_field (PayTech sends it Base64-encoded per docs)
$customFieldRaw = $input['custom_field'] ?? '';
$customDecoded = null;
$invoiceId = null;

if ($customFieldRaw !== '') {
    // Try base64 decode; if it fails, keep raw
    $decoded = base64_decode($customFieldRaw, true);
    if ($decoded !== false) {
        $customDecoded = $decoded;
    } else {
        $customDecoded = $customFieldRaw;
    }

    $json = json_decode($customDecoded, true);
    if (is_array($json) && isset($json['invoice_id'])) {
        $invoiceId = (int)$json['invoice_id'];
    }
}

// Fallback: try parse invoice from ref_command format WHMCS-{invoiceId}-{ts}
if (!$invoiceId && preg_match('/^WHMCS-(\d+)-\d+$/', $refCommand, $m)) {
    $invoiceId = (int)$m[1];
}

if (!$invoiceId) {
    logTransaction($gatewayModuleName, array('input' => $input, 'reason' => 'Missing invoiceId'), 'Error');
    http_response_code(400);
    echo "Missing invoiceId";
    exit;
}

// Validate invoice id exists in WHMCS
checkCbInvoiceID($invoiceId, $gatewayParams['name']);

// Transaction id - use PayTech token if present, else ref_command
$transactionId = $token !== '' ? $token : $refCommand;

// Ensure we don't double-post
checkCbTransID($transactionId);

// Decide action based on type_event
if ($event === 'sale_complete') {
    $paymentAmount = (float)$itemPrice;

    // Apply payment
    addInvoicePayment(
        $invoiceId,
        $transactionId,
        $paymentAmount,
        0.0,
        $gatewayModuleName
    );

    logTransaction($gatewayModuleName, $input, 'Successful');

    http_response_code(200);
    echo "IPN OK";
    exit;
}

if ($event === 'sale_canceled') {
    logTransaction($gatewayModuleName, $input, 'Cancelled');
    http_response_code(200);
    echo "IPN OK";
    exit;
}

// Unknown event: log only
logTransaction($gatewayModuleName, $input, 'Unknown');
http_response_code(200);
echo "IPN OK";
exit;
